import { useState } from 'react'
import { useNavigate } from 'react-router-dom'
import { Calendar, CheckCircle, Sparkles, Hand, Scissors, Package, Flower2, Award, Heart, Clock, TrendingUp, Building2, Star } from 'lucide-react'
import { motion } from 'framer-motion'
import './BookProcedure.css'

const BookProcedure = () => {
  const navigate = useNavigate()
  const [selectedCategory, setSelectedCategory] = useState('')
  const [selectedDate, setSelectedDate] = useState('')
  const [isSubmitted, setIsSubmitted] = useState(false)

  const categories = [
    { 
      id: 'massage', 
      name: 'Массаж', 
      icon: Hand,
      description: 'Расслабляющий и лечебный массаж'
    },
    { 
      id: 'peeling', 
      name: 'Пилинг', 
      icon: Scissors,
      description: 'Глубокое очищение и обновление кожи'
    },
    { 
      id: 'wrap', 
      name: 'Обёртывание', 
      icon: Package,
      description: 'Антицеллюлитное и моделирующее'
    },
    { 
      id: 'aromatherapy', 
      name: 'Ароматерапия', 
      icon: Flower2,
      description: 'Релаксация и восстановление энергии'
    }
  ]

  const salons = [
    { name: 'SpaLux Premium', location: 'Центр', rating: 5, specialties: 'Все процедуры' },
    { name: 'SpaLux Center', location: 'Тверская', rating: 5, specialties: 'Массаж, Пилинг' },
    { name: 'SpaLux Relax', location: 'Арбат', rating: 5, specialties: 'Обёртывание, Ароматерапия' },
    { name: 'SpaLux Wellness', location: 'Сокольники', rating: 5, specialties: 'Все процедуры' }
  ]

  const guarantees = [
    { icon: Heart, title: 'Натуральные компоненты', text: 'Органические средства' },
    { icon: Award, title: 'Опытные мастера', text: 'Сертифицированные специалисты' },
    { icon: Clock, title: 'Гибкий график', text: 'Удобное время' },
    { icon: TrendingUp, title: 'Высокий рейтинг', text: '99% довольных клиентов' }
  ]

  const handleSubmit = (e) => {
    e.preventDefault()
    if (selectedCategory && selectedDate) {
      const savedVisits = localStorage.getItem('spaVisits')
      const visits = savedVisits ? JSON.parse(savedVisits) : []
      
      const categoryData = categories.find(c => c.id === selectedCategory)
      const randomSalon = salons[Math.floor(Math.random() * salons.length)]
      
      const newVisit = {
        id: Date.now(),
        procedure: categoryData.name,
        salon: randomSalon.name,
        date: selectedDate,
        rating: 0
      }

      visits.push(newVisit)
      localStorage.setItem('spaVisits', JSON.stringify(visits))
      
      setIsSubmitted(true)
      setTimeout(() => {
        setIsSubmitted(false)
        setSelectedCategory('')
        setSelectedDate('')
        navigate('/')
      }, 2000)
    }
  }

  const minDate = new Date().toISOString().split('T')[0]
  const maxDate = new Date(Date.now() + 90 * 24 * 60 * 60 * 1000).toISOString().split('T')[0]

  return (
    <div className="book-procedure-page">
      <div className="container">
        <div className="book-header">
          <motion.h1 
            className="book-title"
            initial={{ opacity: 0, x: -20 }}
            animate={{ opacity: 1, x: 0 }}
            transition={{ duration: 0.6 }}
          >
            Записаться на процедуру
          </motion.h1>
          <motion.p 
            className="book-subtitle"
            initial={{ opacity: 0, x: -20 }}
            animate={{ opacity: 1, x: 0 }}
            transition={{ duration: 0.6, delay: 0.1 }}
          >
            Выберите категорию и дату
          </motion.p>
        </div>

        <motion.section 
          className="booking-section"
          initial={{ opacity: 0, y: 20 }}
          animate={{ opacity: 1, y: 0 }}
          transition={{ duration: 0.6, delay: 0.2 }}
        >
          <form className="booking-form" onSubmit={handleSubmit}>
            <div className="form-group">
              <label className="form-label">
                <Sparkles size={18} strokeWidth={2} />
                Категория процедуры
              </label>
              <div className="categories-list">
                {categories.map((category) => {
                  const IconComponent = category.icon
                  return (
                    <motion.button
                      key={category.id}
                      type="button"
                      className={`category-item ${selectedCategory === category.id ? 'selected' : ''}`}
                      onClick={() => setSelectedCategory(category.id)}
                      whileHover={{ x: 5 }}
                      whileTap={{ scale: 0.98 }}
                    >
                      <div className="category-item-icon">
                        <IconComponent size={24} strokeWidth={2} />
                      </div>
                      <div className="category-item-content">
                        <h3 className="category-item-name">{category.name}</h3>
                        <p className="category-item-desc">{category.description}</p>
                      </div>
                    </motion.button>
                  )
                })}
              </div>
            </div>

            <div className="form-group">
              <label htmlFor="date" className="form-label">
                <Calendar size={18} strokeWidth={2} />
                Дата процедуры
              </label>
              <input
                type="date"
                id="date"
                className="form-input"
                value={selectedDate}
                onChange={(e) => setSelectedDate(e.target.value)}
                min={minDate}
                max={maxDate}
                required
              />
            </div>

            <motion.button 
              type="submit" 
              className="submit-button" 
              disabled={!selectedCategory || !selectedDate}
              whileHover={{ y: -2 }}
              whileTap={{ scale: 0.98 }}
            >
              Записаться
            </motion.button>

            {isSubmitted && (
              <motion.div 
                className="success-message"
                initial={{ opacity: 0, y: 10 }}
                animate={{ opacity: 1, y: 0 }}
                transition={{ duration: 0.3 }}
              >
                <CheckCircle size={20} strokeWidth={2} />
                <span>Вы успешно записались на процедуру!</span>
              </motion.div>
            )}
          </form>
        </motion.section>

        <motion.section 
          className="guarantees-section"
          initial={{ opacity: 0, y: 20 }}
          animate={{ opacity: 1, y: 0 }}
          transition={{ duration: 0.6, delay: 0.4 }}
        >
          <h2 className="section-title-minimal">Гарантии</h2>
          <div className="guarantees-grid">
            {guarantees.map((guarantee, index) => {
              const IconComponent = guarantee.icon
              return (
                <motion.div
                  key={guarantee.title}
                  className="guarantee-item"
                  initial={{ opacity: 0, y: 20 }}
                  animate={{ opacity: 1, y: 0 }}
                  transition={{ delay: index * 0.1, duration: 0.5 }}
                  whileHover={{ translateY: -5, borderColor: 'var(--primary)' }}
                >
                  <div className="guarantee-icon">
                    <IconComponent size={24} strokeWidth={2} />
                  </div>
                  <div className="guarantee-content">
                    <h3 className="guarantee-title">{guarantee.title}</h3>
                    <p className="guarantee-text">{guarantee.text}</p>
                  </div>
                </motion.div>
              )
            })}
          </div>
        </motion.section>

        <motion.section 
          className="salons-section"
          initial={{ opacity: 0, y: 20 }}
          animate={{ opacity: 1, y: 0 }}
          transition={{ duration: 0.6, delay: 0.5 }}
        >
          <h2 className="section-title-minimal">Наши салоны</h2>
          <div className="salons-list">
            {salons.map((salon, index) => (
              <motion.div
                key={salon.name}
                className="salon-item"
                initial={{ opacity: 0, scale: 0.9 }}
                animate={{ opacity: 1, scale: 1 }}
                transition={{ delay: index * 0.1, duration: 0.5 }}
                whileHover={{ translateY: -5, borderColor: 'var(--primary)' }}
              >
                <div className="salon-item-left">
                  <div className="salon-item-icon">
                    <Building2 size={24} strokeWidth={2} />
                  </div>
                  <div className="salon-item-content">
                    <h3 className="salon-item-name">{salon.name}</h3>
                    <div className="salon-item-details">
                      <span className="salon-item-location">{salon.location}</span>
                      <div className="salon-item-rating">
                        {Array.from({ length: 5 }, (_, i) => (
                          <Star key={i} size={14} fill="#F59E0B" stroke="#F59E0B" strokeWidth={2} />
                        ))}
                      </div>
                    </div>
                    <p className="salon-item-specialties">{salon.specialties}</p>
                  </div>
                </div>
              </motion.div>
            ))}
          </div>
        </motion.section>
      </div>
    </div>
  )
}

export default BookProcedure

